<?php

/**
 * JNews Tiktok Class
 *
 * @author Jegtheme
 * @since 1.0.0
 * @package jnews-tiktok
 */

namespace JNews\Tiktok;

use JNews\Tiktok\Customizer\Customizer;
use JNews\Tiktok\Element\Register_Elements;
use JNews\Tiktok\Util\JNews_Tiktok_Render;
use JNews\Tiktok\Widget\Register_Widgets;
use JNews\Tiktok\Util\JNews_Tiktok_Api;

/**
 * Class Init
 *
 * @package JNews\Tiktok
 */
class Init {
	/**
	 * @var Init
	 */
	private static $instance;

	/**
	 * Init constructor.
	 */
	private function __construct() {
		$this->setup_init();
		$this->setup_hook();
	}

	/**
	 * @return Init
	 */
	public static function instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}

		return static::$instance;
	}

	/**
	 * Setup Classes
	 */
	private function setup_init() {
		Customizer::instance();
		Register_Elements::instance();
		Register_Widgets::instance();
	}

	/**
	 * Setup Hooks
	 */
	private function setup_hook() {
		add_action( 'init', array( $this, 'custom_rewrite_endpoint' ) );
		add_action('admin_init', array( $this, 'tiktok_callback'));
		add_action( 'admin_enqueue_scripts', array( $this, 'load_localize_script' ), 100 );
		add_action( 'jnews_render_tiktok_feed_footer', array( $this, 'jnews_tiktok_feed_footer' ) );
		add_action( 'wp_print_styles', array( $this, 'load_frontend_css' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'load_admin_asset' ) );
		add_action( 'elementor/editor/after_enqueue_styles', array( $this, 'load_admin_asset' ), 11 );
	}

	/**
	 * Tiktok callback rewrite endpoint
	 */
	public function custom_rewrite_endpoint() {
		add_rewrite_endpoint( 'tiktok-callback', EP_ROOT );
	}

	/**
	 * Tiktok callback
	 */
	public function tiktok_callback() {
		//see iQ4nWlII
		global $pagenow;
		if ( $pagenow === 'customize.php' ) {
			if( isset($_GET['access_token']) && isset($_GET['open_id']) ){
				$tiktok_settings = get_option('jnews_option[jnews_tiktok]');
				if(empty($tiktok_settings) ||  ( !empty($tiktok_settings) && $tiktok_settings['access_token'] !== sanitize_text_field( $_GET['access_token'] )  ) ) {
					//make sure it only fired once if access token is the same to prevent the timestamp of the cache become not accordance with tiktok caching time
					$options = array(
						'access_token' => sanitize_text_field( $_GET['access_token'] ),
						'open_id' => sanitize_text_field( $_GET['open_id'] ),
						'expires_in' => sanitize_text_field( $_GET['expires_in'] ),
						'refresh_token' => sanitize_text_field( $_GET['refresh_token'] ),
						'refresh_expires_in' => sanitize_text_field( $_GET['refresh_expires_in'] ),
					);
				
					update_option( 'jnews_option[jnews_tiktok]', $options );

					$tiktok_data = (new JNews_Tiktok_Api())->get_tiktok_data( sanitize_text_field( $_GET['access_token'] ), sanitize_text_field( $_GET['open_id'] ));
					update_option( 'jnews_tiktok_cached_data', $tiktok_data );
				}
			}
		}
	}

	/**
	 * Load Script
	 */
	public function load_localize_script() {
		if ( is_admin() ) {
			$handle = 'jnews-widget';
			wp_localize_script( $handle, 'jnewstiktokoption', $this->localize_script() );
		}
	}

	/**
	 * Hold localize script
	 *
	 * @return array
	 */
	public function localize_script() {
		return (new JNews_Tiktok_Api())->validation_data();
	}


	public function load_admin_asset() {
		wp_enqueue_style( 'jnews-tiktok-admin', JNEWS_TIKTOK_URL . '/assets/css/admin/admin-style.css', null, JNEWS_TIKTOK_VERSION );
		wp_enqueue_style( 'jnews-tiktok-elementor', JNEWS_TIKTOK_URL . '/assets/css/admin/elementor-backend.css', null, JNEWS_TIKTOK_VERSION );
	}

	/**
	 * Load Frontend CSS
	 */
	public function load_frontend_css() {
		wp_enqueue_style( 'jnews-tiktok', JNEWS_TIKTOK_URL . '/assets/css/frontend.css', null, JNEWS_TIKTOK_VERSION );
	}

	/**
	 * Render Tiktok Feed - Footer
	 */
	public function jnews_tiktok_feed_footer() {
		$option = jnews_get_option( 'tiktok_feed_enable', 'hide' );
		//see iQ4nWlII
		if ( $option === 'show' ) {
			$tiktok_data = (new JNews_Tiktok_Api())->check_cache();
			if($tiktok_data) {
				$param = array(
					'row'      => jnews_get_option( 'footer_tiktok_row', 1 ),
					'column'   => jnews_get_option( 'footer_tiktok_column', 8 ),
					'sort'     => jnews_get_option( 'footer_tiktok_sort_type', 'most_recent' ),
					'hover'    => jnews_get_option( 'footer_tiktok_hover_style', 'zoom' ),
					'open'     => jnews_get_option( 'footer_tiktok_open', true ) ? 'target=\'_blank\'' : '',
					'layout'   => jnews_get_option( 'footer_tiktok_layout', 'masonry' ),
					'button'   => jnews_get_option( 'footer_tiktok_view_button', null ),
					'cover'    => jnews_get_option( 'footer_tiktok_cover', 'cover' ),
				);
				$tiktok = new JNews_Tiktok_Render( $param, $tiktok_data );
				return $tiktok->generate_element();
			} else {
				echo jnews_sanitize_output( '<div class="jeg_empty_module">' . jnews_return_translation( 'You\'re not logged into Tiktok, please login', 'jnews-tiktok', 'tiktok_not_logged_in' ) . '<a href="' . get_site_url() . '/wp-admin/customize.php?autofocus[section]=jnews_tiktok_feed_section">' . jnews_return_translation( ' here', 'jnews-tiktok', 'tiktok_not_logged_in_here' ) .  '</a></div>' );
			}
		}
	}
}
